dnl Process this m4 file to produce 'C' language file.
dnl
dnl If you see this line, you can ignore the next one.
/* Do not edit this file. It is produced from the corresponding .m4 source */
dnl

dnl
dnl DEFINE_GET_FUNC(buftype)
dnl
define(`DEFINE_GET_FUNC',dnl
`dnl
#define GET_FN ncmpi_get_vara_$1_all
')dnl

ifdef(`DATATYPE',
      `ifelse(DATATYPE,
              long long,
              `#define GET_FN ncmpi_get_vara_longlong_all',
              `ifelse(DATATYPE,
                      ,
                      `#define GET_FN ncmpi_get_vara_float_all',
                      `#define GET_FN ncmpi_get_vara_'DATATYPE`_all')')',
      `#define GET_FN ncmpi_get_vara_float_all')


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*   File:         pnetcdf_io.c                                              */
/*                                                                           */
/*   Description:  This program reads data points from a file using PnetCDF  */
/*                                                                           */
/*   Input file format:                                                      */
/*                 netCDF file                                               */
/*                                                                           */
/*   Author:  Wei-keng Liao                                                  */
/*            EECS Department, Northwestern University                       */
/*            email: wkliao@eecs.northwestern.edu                            */
/*                                                                           */
/*   Copyright (C) 2013, Northwestern University                             */
/*   See COPYRIGHT notice in top-level directory.                            */
/*                                                                           */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <mpi.h>
#include <pnetcdf.h>
#include "kmeans.h"

/* Handle errors by printing an error message and exiting with a
 * non-zero status. */
#define ERR {if (err!=NC_NOERR) {printf("Error at line %d file %s: %s\n", __LINE__,__FILE__,ncmpi_strerror(err)); return 0;}}

/*---< pnetcdf_read() >------------------------------------------------------*/
int pnetcdf_read(char        *filename,         /* input netCDF file name */
                 char        *varname,          /* data object name in netCDF */
                 long long   *totalNumObjs_ptr, /* total no. data objects */
                 long long   *numObjs_ptr,      /* no. data objects (local) */
                 int         *numCoords_ptr,    /* no. coordinates */
                 DATATYPE  ***objs_ptr,
                 MPI_Comm     comm)
{
    DATATYPE **objects;
    int        i, rank, nproc;
    int        ncid, ndims, dimids[2], varid, err;
    MPI_Offset len, divd, rem;
    MPI_Offset starts[2], subsizes[2];
    MPI_Offset numObjs, numCoords;

    MPI_Comm_rank(comm, &rank);
    MPI_Comm_size(comm, &nproc);

    err = ncmpi_open(comm, filename, NC_NOWRITE, MPI_INFO_NULL, &ncid);
    ERR

    err = ncmpi_inq_varid(ncid, varname, &varid);
    ERR

    err = ncmpi_inq_varndims(ncid, varid, &ndims);
    ERR
    if (ndims != 2) {
        if (rank == 0)
            printf("Error: read ndims(%d) from netCDF file (should be 2)\n",ndims);
        ncmpi_close(ncid);
        return 0;
    }

    err = ncmpi_inq_vardimid(ncid, varid, dimids);
    ERR

    err = ncmpi_inq_dimlen(ncid, dimids[0], &numObjs);
    ERR
    *totalNumObjs_ptr = numObjs;

    err = ncmpi_inq_dimlen(ncid, dimids[1], &numCoords);
    ERR

    if (_debug) printf("numObjs=%lld numCoords=%lld\n",numObjs,numCoords);

    if (numObjs <= 0 || numCoords <= 0) {
        if (rank == 0)
            printf("Error: file format (%s) numObjs=%lld numCoords=%lld\n",filename,numObjs,numCoords);
        ncmpi_close(ncid);
        MPI_Finalize();
        exit(1);
    }
    if (numObjs < nproc) {
        if (rank == 0)
            printf("Error: number of data points must be larger than the number of MPI processes.\n");
        ncmpi_close(ncid);
        MPI_Finalize();
        exit(1);
    }

    divd = numObjs / nproc;
    rem  = numObjs % nproc;
    len  = (rank < rem) ? rank*(divd+1) : rank*divd + rem;
    starts[0] = len;
    starts[1] = 0;

    /* adjust numObjs to be local size */
    numObjs = (rank < rem) ? divd+1 : divd;

    /* allocate space for data points */
    objects    = (DATATYPE**)malloc(numObjs           * sizeof(DATATYPE*));
    assert(objects != NULL);
    objects[0] = (DATATYPE*) malloc(numObjs*numCoords * sizeof(DATATYPE));
    assert(objects[0] != NULL);
    for (i=1; i<numObjs; i++)
        objects[i] = objects[i-1] + numCoords;

    subsizes[0] = numObjs;
    subsizes[1] = numCoords;

    err = GET_FN(ncid, varid, starts, subsizes, objects[0]);
    ERR

    err = ncmpi_close(ncid);
    ERR

    *numObjs_ptr   = numObjs;
    *numCoords_ptr = numCoords;
    *objs_ptr         = objects;

    return 1;
}

/*---< pnetcdf_read_centers() >----------------------------------------------*/
int pnetcdf_read_centers(char        *filename,    /* input netCDF file name */
                          char        *varname,    /* center variable name */
                          long long    numCenters, /* no. centers  */
                          int          numCoords,  /* no. coordinates */
                          float     ***centers_ptr,
                          MPI_Comm     comm)
{
    float **centers;
    int        i, rank, nproc;
    int        ncid, ndims, dimids[2], varid, err;
    MPI_Offset starts[2], counts[2];
    MPI_Offset in_numObjs, in_numCoords;

    MPI_Comm_rank(comm, &rank);
    MPI_Comm_size(comm, &nproc);

    err = ncmpi_open(comm, filename, NC_NOWRITE, MPI_INFO_NULL, &ncid);
    ERR

    err = ncmpi_inq_varid(ncid, varname, &varid);
    ERR

    err = ncmpi_inq_varndims(ncid, varid, &ndims);
    ERR
    if (ndims != 2) {
        if (rank == 0)
            printf("Error: read ndims(%d) from netCDF file (should be 2)\n",ndims);
        ncmpi_close(ncid);
        return 0;
    }

    err = ncmpi_inq_vardimid(ncid, varid, dimids);
    ERR

    err = ncmpi_inq_dimlen(ncid, dimids[0], &in_numObjs);
    ERR
    if (in_numObjs < numCenters) {
        if (rank == 0)
            printf("Error: number of data points is less than requested\n");
        ncmpi_close(ncid);
        return 0;
    }

    err = ncmpi_inq_dimlen(ncid, dimids[1], &in_numCoords);
    ERR
    if (in_numCoords < numCoords) {
        if (rank == 0)
            printf("Error: dimension size of data points is less than requested\n");
        ncmpi_close(ncid);
        return 0;
    }

    if (_debug) printf("in_numObjs=%lld in_numCoords=%lld\n",in_numObjs,in_numCoords);

    if (numCenters <= 0 || numCoords <= 0) {
        if (rank == 0)
            printf("Error: file (%s) numCenters=%lld numCoords=%d\n",filename,numCenters,numCoords);
        ncmpi_close(ncid);
        return 0;
    }
    if (in_numObjs < nproc) {
        if (rank == 0)
            printf("Error: number of data points must be larger than the number of MPI processes.\n");
        ncmpi_close(ncid);
        return 0;
    }

    starts[0] = 0;
    starts[1] = 0;
    counts[0] = numCenters;
    counts[1] = numCoords;

    /* allocate space for data points */
    centers    = (float**)malloc(numCenters           * sizeof(float*));
    assert(centers != NULL);
    centers[0] = (float*) malloc(numCenters*numCoords * sizeof(float));
    assert(centers[0] != NULL);
    for (i=1; i<numCenters; i++)
        centers[i] = centers[i-1] + numCoords;

    err = ncmpi_get_vara_float_all(ncid, varid, starts, counts, centers[0]);
    ERR

    err = ncmpi_close(ncid);
    ERR

    *centers_ptr = centers;

    return 1;
}


/*---< pnetcdf_write() >-----------------------------------------------------*/
int pnetcdf_write(char      *outFileName,  /* output file name */
                  long long  numClusters,  /* no. clusters */
                  long long  numObjs,      /* no. data objects */
                  int        numCoords,    /* no. coordinates (local) */
                  float    **clusters,     /* [numClusters][numCoords] centers*/
                  long long *membership,   /* [numObjs] */
                  long long  totalNumObjs, /* total no. data objects */
                  MPI_Comm   comm,
                  int        verbose)
{
    int  rank, nproc, cmode;
    int  ncid, dimids[3], clusters_vid, membership_vid, err;
    MPI_Offset start, count, divd, rem;

    MPI_Comm_rank(comm, &rank);
    MPI_Comm_size(comm, &nproc);

    /* output: the coordinates of the cluster centres ----------------------*/
    /* only proc 0 matters this, because clusters[] are the same across all
       proc */

    /* to save the results in a new netCDF file */
    if (rank == 0 && verbose) {
        printf("Writing coordinates of K=%lld cluster centers to file \"%s\"\n",
               numClusters, outFileName);

        printf("Writing membership of N=%lld data objects to file \"%s\"\n",
               totalNumObjs, outFileName);
    }

    /* Create the file. The NC_CLOBBER parameter tells netCDF to
     * overwrite this file, if it already exists.*/
    cmode = NC_CLOBBER | NC_64BIT_DATA;
    err = ncmpi_create(comm, outFileName, cmode, MPI_INFO_NULL, &ncid);
    ERR

    /* Define the dimensions. NetCDF will hand back an ID for each. */
    err = ncmpi_def_dim(ncid, "num_clusters", numClusters, &dimids[0]);
    ERR

    err = ncmpi_def_dim(ncid, "num_coordinates", numCoords, &dimids[1]);
    ERR

    err = ncmpi_def_dim(ncid, "num_elements", totalNumObjs, &dimids[2]);
    ERR

    /* Define the clusters variable. The type of the variable in this case is
     * NC_FLOAT (4-byte float). */
    err = ncmpi_def_var(ncid, "clusters", NC_FLOAT, 2, dimids, &clusters_vid);
    ERR

    /* Define the membership variable. The type of the variable in this case is
     * NC_INT (4-byte integer). */
    err = ncmpi_def_var(ncid, "membership", NC_INT64, 1, &dimids[2], &membership_vid);
    ERR

    /* End define mode. This tells netCDF we are done defining metadata */
    err = ncmpi_enddef(ncid);
    ERR

    /* write cluster centers */
    err = ncmpi_put_var_float_all(ncid, clusters_vid, *clusters);
    ERR

    /* write membership variable */
    divd  = totalNumObjs / nproc;
    rem   = totalNumObjs % nproc;
    start = (rank < rem) ? rank*(divd+1) : rank*divd + rem;
    count = numObjs;
    if (_debug) printf("%2d: start=%lld count=%lld\n",rank,start,count);

    err = ncmpi_put_vara_longlong_all(ncid, membership_vid, &start, &count,
                                      membership);
    ERR

    /* Close the file. This frees up any internal netCDF resources
     * associated with the file, and flushes any buffers. */
    err = ncmpi_close(ncid);
    ERR

    return 1;
}

